<?php
// *** PROCESAMIENTO PHP - GESTIÓN DE USUARIOS ***
// Incluido por index.php, $pdo y $_SESSION disponibles.

// --- VERIFICACIÓN DE ROL ---
// Solo administradores pueden acceder a esta página
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'administrador') {
    // Mostrar error o redirigir a home
    echo '<div class="alert alert-danger">Acceso denegado. Permisos insuficientes.</div>';
    // Podrías hacer: header('Location: index.php?page=home'); exit; pero puede dar error si ya hubo output
    return; // Detener la ejecución de este script incluido
}

$mensaje_exito = '';
$mensaje_error = '';
$usuario_a_editar = null;
$modo_edicion = false;

// --- Mensajes de éxito/error por GET ---
if (isset($_GET['success'])) {
    switch ($_GET['success']) {
        case 'created': $mensaje_exito = "¡Usuario creado exitosamente!"; break;
        case 'updated': $mensaje_exito = "¡Usuario actualizado exitosamente!"; break;
        case 'deleted': $mensaje_exito = "¡Usuario eliminado exitosamente!"; break;
    }
}
if (isset($_GET['error'])) {
     $mensaje_error = "Ocurrió un error: " . htmlspecialchars(urldecode($_GET['error']));
}


// --- DETECTAR MODO EDICIÓN ---
if (isset($_GET['edit_id'])) {
    $id_para_editar = filter_input(INPUT_GET, 'edit_id', FILTER_VALIDATE_INT);
    if ($id_para_editar) {
        try {
            $sql_edit = "SELECT id, nombre_usuario, nombre_completo, rol, activo FROM usuarios WHERE id = ?";
            $q_edit = $pdo->prepare($sql_edit);
            $q_edit->execute([$id_para_editar]);
            $usuario_a_editar = $q_edit->fetch(PDO::FETCH_ASSOC);
            if ($usuario_a_editar) {
                $modo_edicion = true;
            } else {
                header("Location: index.php?page=usuarios&error=" . urlencode("Usuario no encontrado.")); exit;
            }
        } catch (PDOException $e) {
            error_log("Error al cargar usuario para editar: " . $e->getMessage());
            $mensaje_error = "Error al cargar datos del usuario.";
        }
    } else {
         header("Location: index.php?page=usuarios&error=" . urlencode("ID de usuario inválido.")); exit;
    }
}


// --- PROCESAR ELIMINACIÓN ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_id']) && isset($_POST['delete_type']) && $_POST['delete_type'] === 'usuario') {
    $id_para_eliminar = filter_input(INPUT_POST, 'delete_id', FILTER_VALIDATE_INT);
    $error_url_param = '';

    if ($id_para_eliminar) {
        // *** PROTECCIÓN IMPORTANTE: No permitir eliminar al usuario actual ***
        if ($id_para_eliminar == $_SESSION['user_id']) {
             $error_url_param = urlencode("No puedes eliminar tu propia cuenta de usuario.");
        } else {
             // (Opcional) Protección extra: No permitir eliminar al último administrador
             $sql_count_admin = "SELECT COUNT(*) FROM usuarios WHERE rol = 'administrador' AND activo = 1";
             $q_count = $pdo->query($sql_count_admin);
             $num_admins = $q_count->fetchColumn();

             $sql_is_admin = "SELECT rol FROM usuarios WHERE id = ?";
             $q_is_admin = $pdo->prepare($sql_is_admin);
             $q_is_admin->execute([$id_para_eliminar]);
             $rol_a_eliminar = $q_is_admin->fetchColumn();

             if ($rol_a_eliminar === 'administrador' && $num_admins <= 1) {
                 $error_url_param = urlencode("No se puede eliminar al último administrador activo.");
             } else {
                 // Proceder con la eliminación
                 try {
                    $sql_delete = "DELETE FROM usuarios WHERE id = ?";
                    $q_delete = $pdo->prepare($sql_delete);
                    $q_delete->execute([$id_para_eliminar]);

                    if ($q_delete->rowCount() > 0) {
                        header("Location: index.php?page=usuarios&success=deleted"); exit;
                    } else {
                        $error_url_param = urlencode("No se encontró el usuario para eliminar (ID: $id_para_eliminar).");
                    }
                } catch (PDOException $e) {
                    error_log("Error PDO al eliminar usuario: " . $e->getMessage());
                    // Podría fallar si tiene registros de votos asociados y no hay ON DELETE CASCADE/SET NULL
                    $error_url_param = urlencode("Error de base de datos al eliminar. Verifique dependencias.");
                }
             }
        }
    } else {
        $error_url_param = urlencode("ID inválido para eliminar.");
    }
    if ($error_url_param) {
        header("Location: index.php?page=usuarios&error=" . $error_url_param); exit;
    }
}


// --- Procesar Envío del Formulario (Crear o Actualizar Usuario) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['guardar_usuario'])) {
    $id_usuario_form = filter_input(INPUT_POST, 'id_usuario', FILTER_VALIDATE_INT);
    $nombre_usuario = trim($_POST['nombre_usuario'] ?? '');
    $nombre_completo = trim($_POST['nombre_completo'] ?? '');
    $rol = $_POST['rol'] ?? 'registrador'; // Rol por defecto
    $activo = isset($_POST['activo']) ? 1 : 0;
    $password = $_POST['password'] ?? '';
    $password_confirm = $_POST['password_confirm'] ?? '';
    $update_password = false; // Flag para saber si actualizaremos la contraseña

    // Validaciones
    if (empty($nombre_usuario)) {
        $mensaje_error = "El nombre de usuario es obligatorio.";
    } elseif (!in_array($rol, ['administrador', 'registrador'])) {
        $mensaje_error = "El rol seleccionado no es válido.";
    } elseif (!$id_usuario_form && empty($password)) { // Contraseña obligatoria al crear
        $mensaje_error = "La contraseña es obligatoria para nuevos usuarios.";
    } elseif (!empty($password) && $password !== $password_confirm) { // Si se ingresó pass, verificar confirmación
        $mensaje_error = "Las contraseñas no coinciden.";
    } elseif (!empty($password) && strlen($password) < 6) { // Validación básica de longitud
         $mensaje_error = "La contraseña debe tener al menos 6 caracteres.";
    } else {
        // Si se proporcionó una contraseña (ya sea creando o editando), la procesaremos
        if (!empty($password)) {
            $password_hash = password_hash($password, PASSWORD_DEFAULT);
            if ($password_hash === false) {
                $mensaje_error = "Error al generar el hash de la contraseña.";
            } else {
                $update_password = true; // Marcar para incluir en el UPDATE
            }
        }

        if (empty($mensaje_error)) {
            try {
                if ($id_usuario_form) { // --- ACTUALIZAR ---
                    $sql_params = [$nombre_usuario, $nombre_completo ?: null, $rol, $activo];
                    $sql = "UPDATE usuarios SET nombre_usuario = ?, nombre_completo = ?, rol = ?, activo = ?";
                    if ($update_password) {
                        $sql .= ", password = ?";
                        $sql_params[] = $password_hash;
                    }
                    $sql .= ", updated_at = NOW() WHERE id = ?";
                    $sql_params[] = $id_usuario_form;

                    $q = $pdo->prepare($sql);
                    if ($q->execute($sql_params)) {
                        header("Location: index.php?page=usuarios&success=updated"); exit;
                    } else {
                        $mensaje_error = "Error al actualizar el usuario.";
                    }

                } else { // --- CREAR ---
                    $sql = "INSERT INTO usuarios (nombre_usuario, password, rol, nombre_completo, activo, created_at, updated_at)
                            VALUES (?, ?, ?, ?, ?, NOW(), NOW())";
                    $q = $pdo->prepare($sql);
                    // Crear necesita el hash obligatoriamente
                    if ($update_password && $q->execute([$nombre_usuario, $password_hash, $rol, $nombre_completo ?: null, $activo])) {
                        header("Location: index.php?page=usuarios&success=created"); exit;
                    } else {
                         // Si $update_password es false, hubo error en hash o validación previa
                         $mensaje_error = $mensaje_error ?: "Error al crear el usuario.";
                    }
                }
            } catch (PDOException $e) {
                error_log("Error DB usuario: " . $e->getMessage());
                if ($e->getCode() == '23000') { // Error de constraint unique (nombre_usuario)
                     $mensaje_error = "Error: El nombre de usuario '$nombre_usuario' ya está en uso.";
                } else {
                     $mensaje_error = "Error de base de datos al " . ($id_usuario_form ? 'actualizar' : 'crear') . " el usuario.";
                }
            }
        }
    }
    // Si hubo error, asegurar que seguimos en modo edición si aplica
    if (!empty($mensaje_error) && $id_usuario_form) {
         $modo_edicion = true;
         // Recargar datos originales excepto los del post que causaron error
          $sql_reload = "SELECT id, nombre_usuario, nombre_completo, rol, activo FROM usuarios WHERE id = ?";
          $q_reload = $pdo->prepare($sql_reload); $q_reload->execute([$id_usuario_form]);
          $usuario_a_editar = $q_reload->fetch(PDO::FETCH_ASSOC);
          // Sobrescribir con los valores del post para mantener el intento del usuario
          $usuario_a_editar['nombre_usuario'] = $_POST['nombre_usuario'] ?? $usuario_a_editar['nombre_usuario'];
          $usuario_a_editar['nombre_completo'] = $_POST['nombre_completo'] ?? $usuario_a_editar['nombre_completo'];
          $usuario_a_editar['rol'] = $_POST['rol'] ?? $usuario_a_editar['rol'];
          $usuario_a_editar['activo'] = isset($_POST['activo']) ? 1 : 0; // Mantener estado del checkbox
    }
}


// --- Cargar Lista de Usuarios (Leer) ---
$lista_usuarios = [];
$error_carga_lista = null;
try {
    $sql_select = "SELECT id, nombre_usuario, nombre_completo, rol, activo, created_at FROM usuarios ORDER BY nombre_usuario";
    $q_select = $pdo->query($sql_select); // No necesita prepare si no hay parámetros
    $lista_usuarios = $q_select->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Error al cargar lista usuarios: " . $e->getMessage());
    $error_carga_lista = "Error al cargar la lista de usuarios.";
}

?>

<div class="container-fluid">
    <h1 class="mb-4"><i class="fas fa-user-cog"></i> Gestión de Usuarios</h1>

    <!-- Área de mensajes -->
    <?php if ($mensaje_exito): ?><div class="alert alert-success alert-dismissible fade show" role="alert"><?php echo htmlspecialchars($mensaje_exito); ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div><?php endif; ?>
    <?php if ($mensaje_error): ?><div class="alert alert-danger alert-dismissible fade show" role="alert"><?php echo htmlspecialchars($mensaje_error); ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div><?php endif; ?>
    <?php if ($error_carga_lista): ?><div class="alert alert-warning"><?php echo htmlspecialchars($error_carga_lista); ?></div><?php endif; ?>

    <!-- Formulario para agregar/editar -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary"><?php echo $modo_edicion ? 'Editar Usuario' : 'Registrar Nuevo Usuario'; ?></h6>
        </div>
        <div class="card-body">
            <form action="index.php?page=usuarios<?php echo $modo_edicion ? '&edit_id='.htmlspecialchars($usuario_a_editar['id']) : ''; ?>" method="POST">
                <?php if ($modo_edicion): ?>
                    <input type="hidden" name="id_usuario" value="<?php echo htmlspecialchars($usuario_a_editar['id']); ?>">
                <?php endif; ?>

                <div class="row mb-3">
                    <div class="col-md-6">
                        <label for="nombre_usuario" class="form-label">Nombre de Usuario: <span class="text-danger">*</span></label>
                        <input type="text" class="form-control <?php echo (strpos($mensaje_error, 'usuario') !== false) ? 'is-invalid' : ''; ?>" id="nombre_usuario" name="nombre_usuario" required value="<?php echo htmlspecialchars($usuario_a_editar['nombre_usuario'] ?? ($_POST['nombre_usuario'] ?? '')); ?>">
                         <?php if (strpos($mensaje_error, 'usuario') !== false): ?><div class="invalid-feedback"><?php echo htmlspecialchars($mensaje_error); ?></div><?php endif; ?>
                    </div>
                     <div class="col-md-6">
                        <label for="nombre_completo" class="form-label">Nombre Completo:</label>
                        <input type="text" class="form-control" id="nombre_completo" name="nombre_completo" value="<?php echo htmlspecialchars($usuario_a_editar['nombre_completo'] ?? ($_POST['nombre_completo'] ?? '')); ?>">
                    </div>
                </div>

                 <div class="row mb-3">
                     <div class="col-md-6">
                        <label for="rol" class="form-label">Rol: <span class="text-danger">*</span></label>
                        <select class="form-select <?php echo (strpos($mensaje_error, 'rol') !== false) ? 'is-invalid' : ''; ?>" id="rol" name="rol" required>
                            <option value="registrador" <?php echo (($usuario_a_editar['rol'] ?? ($_POST['rol'] ?? '')) == 'registrador') ? 'selected' : ''; ?>>Registrador</option>
                            <option value="administrador" <?php echo (($usuario_a_editar['rol'] ?? ($_POST['rol'] ?? '')) == 'administrador') ? 'selected' : ''; ?>>Administrador</option>
                        </select>
                         <?php if (strpos($mensaje_error, 'rol') !== false): ?><div class="invalid-feedback"><?php echo htmlspecialchars($mensaje_error); ?></div><?php endif; ?>
                    </div>
                    <div class="col-md-6 d-flex align-items-center pt-4">
                         <div class="form-check form-switch">
                            <input class="form-check-input" type="checkbox" role="switch" id="activo" name="activo" value="1" <?php echo ($modo_edicion ? ($usuario_a_editar['activo'] ? 'checked' : '') : 'checked'); ?>>
                            <label class="form-check-label" for="activo">Usuario Activo</label>
                        </div>
                    </div>
                 </div>

                 <hr>
                 <p class="text-muted small"><?php echo $modo_edicion ? 'Ingrese una nueva contraseña solo si desea cambiarla (mínimo 6 caracteres).' : 'Establezca la contraseña para el nuevo usuario (mínimo 6 caracteres).' ?></p>

                <div class="row mb-3">
                    <div class="col-md-6">
                         <label for="password" class="form-label">Contraseña: <?php if(!$modo_edicion): ?><span class="text-danger">*</span><?php endif; ?></label>
                        <input type="password" class="form-control <?php echo (strpos($mensaje_error, 'ontraseña') !== false || strpos($mensaje_error, 'caracteres') !== false) ? 'is-invalid' : ''; ?>" id="password" name="password" <?php echo !$modo_edicion ? 'required' : ''; ?> minlength="6">
                        <?php if (strpos($mensaje_error, 'ontraseña') !== false || strpos($mensaje_error, 'caracteres') !== false): ?><div class="invalid-feedback"><?php echo htmlspecialchars($mensaje_error); ?></div><?php endif; ?>
                    </div>
                     <div class="col-md-6">
                         <label for="password_confirm" class="form-label">Confirmar Contraseña: <?php if(!$modo_edicion): ?><span class="text-danger">*</span><?php endif; ?></label>
                        <input type="password" class="form-control <?php echo (strpos($mensaje_error, 'oinciden') !== false) ? 'is-invalid' : ''; ?>" id="password_confirm" name="password_confirm" <?php echo !$modo_edicion ? 'required' : ''; ?> minlength="6">
                         <?php if (strpos($mensaje_error, 'oinciden') !== false): ?><div class="invalid-feedback">Las contraseñas no coinciden.</div><?php endif; ?>
                    </div>
                </div>


                <input type="hidden" name="guardar_usuario" value="1">
                 <button type="submit" class="btn btn-primary">
                    <i class="fas fa-<?php echo $modo_edicion ? 'sync-alt' : 'save'; ?>"></i> <?php echo $modo_edicion ? 'Actualizar' : 'Crear'; ?> Usuario
                </button>
                <?php if ($modo_edicion): ?>
                <a href="index.php?page=usuarios" class="btn btn-secondary"><i class="fas fa-times"></i> Cancelar Edición</a>
                <?php else: ?>
                <button type="reset" class="btn btn-secondary"><i class="fas fa-eraser"></i> Limpiar</button>
                <?php endif; ?>
            </form>
        </div>
    </div>

     <!-- Tabla para listar usuarios -->
    <div class="card shadow mb-4">
        <div class="card-header py-3"><h6 class="m-0 font-weight-bold text-primary">Lista de Usuarios Registrados</h6></div>
        <div class="card-body">
            <?php if (empty($lista_usuarios) && empty($error_carga_lista)): ?>
                <p class="text-center">No hay usuarios registrados.</p>
            <?php elseif (!empty($lista_usuarios)): ?>
                <div class="table-responsive">
                    <table class="table table-bordered table-hover table-sm" width="100%" cellspacing="0">
                        <thead class="table-light">
                            <tr>
                                <th>ID</th><th>Usuario</th><th>Nombre Completo</th><th>Rol</th>
                                <th class="text-center">Activo</th><th>Creado</th><th class="text-center">Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($lista_usuarios as $usuario):
                                $nombre_usuario_tabla = htmlspecialchars($usuario['nombre_usuario']);
                            ?>
                            <tr>
                                <td><?php echo htmlspecialchars($usuario['id']); ?></td>
                                <td><?php echo $nombre_usuario_tabla; ?></td>
                                <td><?php echo htmlspecialchars($usuario['nombre_completo'] ?: '-'); ?></td>
                                <td><?php echo ucfirst(htmlspecialchars($usuario['rol'])); ?></td>
                                <td class="text-center">
                                    <?php if ($usuario['activo']): ?>
                                        <span class="badge bg-success">Sí</span>
                                    <?php else: ?>
                                        <span class="badge bg-secondary">No</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo htmlspecialchars(date('d/m/y H:i', strtotime($usuario['created_at']))); ?></td>
                                <td class="text-center align-middle">
                                    <a href="index.php?page=usuarios&edit_id=<?php echo htmlspecialchars($usuario['id']); ?>" class="btn btn-sm btn-warning me-1 py-0 px-1" title="Editar Usuario">
                                        <i class="fas fa-edit fa-xs"></i>
                                    </a>
                                    <?php // Deshabilitar botón de borrar para el usuario actual ?>
                                    <button type="button" class="btn btn-sm btn-danger py-0 px-1" title="Eliminar Usuario"
                                            data-bs-toggle="modal" data-bs-target="#confirmDeleteModal"
                                            data-id="<?php echo htmlspecialchars($usuario['id']); ?>"
                                            data-name="<?php echo $nombre_usuario_tabla; ?>"
                                            data-type="usuario"
                                            data-page="usuarios"
                                            <?php echo ($usuario['id'] == $_SESSION['user_id']) ? 'disabled' : ''; ?>>
                                        <i class="fas fa-trash fa-xs"></i>
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>

</div> <!-- Fin container-fluid -->