<?php
// *** PROCESAMIENTO PHP ***
// Incluido por index.php, $pdo y $_SESSION disponibles.

$mensaje_error = '';
$resultados_votos = []; // Para la tabla y datos base del gráfico
$total_general_votos = 0;
$chart_labels = []; // Etiquetas para el gráfico (Nombres)
$chart_data = [];   // Valores para el gráfico (Votos)
$chart_background_colors = []; // Colores para las barras
$id_eleccion_actual = DEFAULT_ELECTION_ID; // Asumimos elección 1 por ahora

// --- Carga de Datos desde la BD ---
try {
    // Consulta optimizada para obtener resultados agregados por candidato
    $sql = "SELECT
                c.id AS candidato_id, -- Incluir ID por si acaso
                c.nombre,
                c.apellido,
                c.partido_politico,
                c.foto_url,
                SUM(v.cantidad_votos) AS total_votos
            FROM votos AS v
            JOIN candidatos AS c ON v.id_candidato = c.id
            WHERE v.id_eleccion = ?  -- Filtrar por la elección activa
            GROUP BY c.id, c.nombre, c.apellido, c.partido_politico, c.foto_url -- Agrupar por todos los campos del candidato
            ORDER BY total_votos DESC";

    $q = $pdo->prepare($sql);
    $q->execute([$id_eleccion_actual]);
    $resultados_db = $q->fetchAll(PDO::FETCH_ASSOC);

    if (empty($resultados_db)) {
        $mensaje_error = "Aún no hay votos registrados para la elección ID: $id_eleccion_actual.";
    } else {
        $resultados_votos = $resultados_db; // Guardar resultados para la tabla

        // Procesamiento para el gráfico y la tabla
        $colores_base = ['#36a2eb', '#ff6384', '#ffce56', '#4bc0c0', '#9966ff', '#ff9f40', '#ff63a0', '#c45850']; // Más colores base
        $color_index = 0;

        foreach ($resultados_votos as $resultado) {
            $nombre_completo_chart = $resultado['nombre'] . ' ' . $resultado['apellido'];
            $total_general_votos += $resultado['total_votos']; // Sumar al total general
            $chart_labels[] = $nombre_completo_chart;    // Etiqueta para el gráfico
            $chart_data[] = $resultado['total_votos'];          // Valor para el gráfico
            // Asignar color (rotando si hay más candidatos que colores base)
            $chart_background_colors[] = $colores_base[$color_index % count($colores_base)];
            $color_index++;
        }
    }

} catch (PDOException $e) {
    error_log("Error al generar reporte de votos: " . $e->getMessage());
    $mensaje_error = "Error al consultar los resultados de la base de datos.";
}

?>

<div class="container-fluid">
    <h1 class="mb-4"><i class="fas fa-chart-bar"></i> Reporte de Votos (Elección ID: <?php echo $id_eleccion_actual; ?>)</h1>

    <!-- Área de mensajes de error -->
    <?php if ($mensaje_error): ?>
        <div class="alert alert-warning" role="alert">
            <?php echo htmlspecialchars($mensaje_error); ?>
        </div>
    <?php elseif (!empty($resultados_votos)): ?>
        <!-- Mostrar solo si hay datos -->

        <!-- Opciones de Exportación -->
        <div class="mb-3 text-end">
            <a href="export_report.php?format=csv&election_id=<?php echo $id_eleccion_actual; ?>" class="btn btn-sm btn-success" target="_blank">
                <i class="fas fa-file-csv"></i> Exportar a CSV (Excel)
            </a>
            <!-- Enlace PDF comentado (no implementado) -->
            <!-- <a href="export_report.php?format=pdf&election_id=<?php //echo $id_eleccion_actual; ?>" class="btn btn-sm btn-danger" target="_blank">
                <i class="fas fa-file-pdf"></i> Exportar a PDF
            </a> -->
        </div>

        <!-- 1. Gráfico de Barras -->
        <div class="card shadow mb-4">
            <div class="card-header py-3">
                <h6 class="m-0 font-weight-bold text-primary">Resultados Gráficos por Candidato</h6>
            </div>
            <div class="card-body">
                 <?php if (!empty($chart_data)): // Solo mostrar si hay datos para el gráfico ?>
                    <div style="height: 400px; position: relative;">
                        <canvas id="resultsChart"></canvas>
                    </div>
                    <!-- Leyenda de Imágenes -->
                    <div class="mt-4 text-center d-flex flex-wrap justify-content-center align-items-center">
                        <?php foreach ($resultados_votos as $index => $resultado): ?>
                            <div class="text-center mx-2 mb-2 p-1 border rounded" style="background-color: <?php echo $chart_background_colors[$index]; ?>; display: inline-flex; align-items: center;">
                                 <img src="<?php echo !empty($resultado['foto_url']) ? htmlspecialchars($resultado['foto_url']) : 'assets/placeholder.png'; ?>"
                                     alt="Logo" class="img-thumbnail me-2" width="35" height="35" style="object-fit: contain; background: white;"
                                     onerror="this.onerror=null; this.src='assets/placeholder.png';">
                                 <small class="d-block text-white" style="font-size: 0.8em; text-shadow: 1px 1px 2px black;">
                                    <?php // Puedes poner el nombre aquí si quieres, aunque puede ser redundante con la tabla
                                        // echo htmlspecialchars($resultado['nombre'] . ' ' . $resultado['apellido']);
                                    ?>
                                 </small>
                            </div>
                        <?php endforeach; ?>
                    </div>
                 <?php else: ?>
                    <p class="text-center text-muted">No hay datos suficientes para generar el gráfico.</p>
                 <?php endif; ?>
            </div>
        </div>

        <!-- 2. Tabla de Resultados con Porcentajes -->
        <div class="card shadow mb-4">
            <div class="card-header py-3">
                <h6 class="m-0 font-weight-bold text-primary">Resultados Detallados por Candidato</h6>
            </div>
            <div class="card-body">
                <p class="text-end"><strong>Total General de Votos Registrados: <?php echo number_format($total_general_votos); ?></strong></p>
                <div class="table-responsive">
                    <table class="table table-bordered table-hover table-sm" id="dataTableResultados" width="100%" cellspacing="0">
                        <thead class="table-light">
                            <tr>
                                <th class="text-center" style="width: 80px;">Logo</th>
                                <th>Candidato</th>
                                <th>Partido Político</th>
                                <th class="text-end">Votos Totales</th>
                                <th class="text-end" style="min-width: 150px;">Porcentaje (%)</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($resultados_votos as $resultado): ?>
                                <?php
                                // Calcular porcentaje (evitando división por cero)
                                $porcentaje = ($total_general_votos > 0)
                                              ? ($resultado['total_votos'] / $total_general_votos) * 100
                                              : 0;
                                $nombre_completo = $resultado['nombre'] . ' ' . $resultado['apellido'];
                                ?>
                                <tr>
                                    <td class="text-center align-middle">
                                        <img src="<?php echo !empty($resultado['foto_url']) ? htmlspecialchars($resultado['foto_url']) : 'assets/placeholder.png'; ?>"
                                             alt="Logo" class="img-thumbnail" width="50" height="50" style="object-fit: contain;"
                                              onerror="this.onerror=null; this.src='assets/placeholder.png';">
                                    </td>
                                    <td class="align-middle"><?php echo htmlspecialchars($nombre_completo); ?></td>
                                    <td class="align-middle"><?php echo htmlspecialchars($resultado['partido_politico'] ?: 'Independiente'); ?></td>
                                    <td class="text-end align-middle"><?php echo number_format($resultado['total_votos']); ?></td>
                                    <td class="text-end align-middle">
                                        <div class="progress" style="height: 20px; font-size: 0.8rem;" title="<?php echo number_format($porcentaje, 2); ?>%">
                                            <div class="progress-bar" role="progressbar"
                                                 style="width: <?php echo number_format($porcentaje, 2); ?>%;"
                                                 aria-valuenow="<?php echo number_format($porcentaje, 2); ?>"
                                                 aria-valuemin="0" aria-valuemax="100">
                                                <?php echo number_format($porcentaje, 2); ?>%
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                         <tfoot class="table-light fw-bold">
                             <tr>
                                 <td colspan="3" class="text-end">TOTAL</td>
                                 <td class="text-end"><?php echo number_format($total_general_votos); ?></td>
                                 <td class="text-end">100.00%</td>
                             </tr>
                         </tfoot>
                    </table>
                </div>
            </div>
        </div>

    <?php endif; // Fin del else (si hay datos) ?>

</div> <!-- Fin container-fluid -->

<!-- Incluir Chart.js (solo si hay datos para mostrar) -->
<?php if (!empty($chart_data)): ?>
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
<script>
    document.addEventListener('DOMContentLoaded', function () {
        const canvas = document.getElementById('resultsChart');
        if (canvas) {
            const ctx = canvas.getContext('2d');
            // Datos para el gráfico (pasados desde PHP)
            const labels = <?php echo json_encode($chart_labels); ?>;
            const dataValues = <?php echo json_encode($chart_data); ?>;
            const backgroundColors = <?php echo json_encode($chart_background_colors); ?>;

            new Chart(ctx, {
                type: 'bar', // Tipo de gráfico: barras verticales
                data: {
                    labels: labels, // Nombres de los candidatos
                    datasets: [{
                        label: 'Total Votos', // Etiqueta para la leyenda del dataset
                        data: dataValues,     // Los valores (votos)
                        backgroundColor: backgroundColors, // Colores de las barras
                        borderColor: backgroundColors.map(color => color.replace('0.8', '1')), // Borde un poco más opaco
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,       // Hacer el gráfico responsivo
                    maintainAspectRatio: false, // No mantener aspect ratio para llenar el contenedor
                    plugins: {
                        legend: {
                            display: false // Ocultar la leyenda automática (usamos la de imágenes)
                        },
                        title: {
                            display: true,
                            text: 'Distribución de Votos por Candidato',
                            font: { size: 16 }, // Tamaño ajustado
                            padding: { top: 10, bottom: 20 }
                        },
                        tooltip: {
                             callbacks: {
                                label: function(context) {
                                    let label = context.dataset.label || '';
                                    if (label) { label += ': '; }
                                    if (context.parsed.y !== null) {
                                        // Formatear número con separadores de miles
                                        label += new Intl.NumberFormat('es-ES').format(context.parsed.y);
                                    }
                                    return label;
                                },
                                title: function(context) { return context[0].label; }
                            }
                        }
                    },
                    scales: {
                        y: { // Eje Y (Votos)
                            beginAtZero: true,
                            title: { display: true, text: 'Número de Votos' }
                        },
                        x: { // Eje X (Candidatos)
                             title: { display: false }, // Ocultar título eje X
                             ticks: {
                                 autoSkip: false, // No saltar etiquetas automáticamente
                                 maxRotation: 70, // Rotar etiquetas si son largas
                                 minRotation: 45
                             }
                        }
                    }
                }
            });
        }
    });
</script>
<?php endif; // Fin del if (!empty($chart_data)) ?>