<?php
// *** PROCESAMIENTO PHP ***
// Incluido por index.php, $pdo y $_SESSION disponibles.

$mensaje_exito = '';
$mensaje_error = '';
$id_eleccion_actual = DEFAULT_ELECTION_ID; // Usamos la elección por defecto
$candidato_a_editar = null; // Variable para guardar datos si estamos editando
$modo_edicion = false; // Flag para saber si estamos editando

// --- Mensajes de éxito/error por GET (tras redirección) ---
if (isset($_GET['success'])) {
    switch ($_GET['success']) {
        case 'created': $mensaje_exito = "¡Candidato registrado exitosamente!"; break;
        case 'updated': $mensaje_exito = "¡Candidato actualizado exitosamente!"; break;
        case 'deleted': $mensaje_exito = "¡Candidato eliminado exitosamente!"; break;
    }
}
if (isset($_GET['error'])) {
     $mensaje_error = "Ocurrió un error en la operación anterior: " . htmlspecialchars(urldecode($_GET['error']));
}


// --- DETECTAR MODO EDICIÓN (CARGAR DATOS) ---
if (isset($_GET['edit_id'])) {
    $id_para_editar = filter_input(INPUT_GET, 'edit_id', FILTER_VALIDATE_INT);
    if ($id_para_editar) {
        try {
            $sql_edit = "SELECT id, nombre, apellido, partido_politico, foto_url FROM candidatos WHERE id = ? AND id_eleccion = ?";
            $q_edit = $pdo->prepare($sql_edit);
            $q_edit->execute([$id_para_editar, $id_eleccion_actual]);
            $candidato_a_editar = $q_edit->fetch(PDO::FETCH_ASSOC);
            if ($candidato_a_editar) {
                $modo_edicion = true; // Activar modo edición
            } else {
                // Usar redirección con mensaje de error
                header("Location: index.php?page=candidatos&error=" . urlencode("Candidato no encontrado o no pertenece a esta elección."));
                exit;
            }
        } catch (PDOException $e) {
            error_log("Error al cargar candidato para editar: " . $e->getMessage());
            $mensaje_error = "Error al cargar datos del candidato para edición."; // Mostrar error en la página actual
        }
    } else {
         header("Location: index.php?page=candidatos&error=" . urlencode("ID de candidato inválido para editar."));
         exit;
    }
}

// --- PROCESAR ELIMINACIÓN ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_id']) && isset($_POST['delete_type']) && $_POST['delete_type'] === 'candidato') {
    $id_para_eliminar = filter_input(INPUT_POST, 'delete_id', FILTER_VALIDATE_INT);
    $error_url_param = ''; // Para la redirección en caso de error

    if ($id_para_eliminar) {
        try {
            $pdo->beginTransaction(); // Iniciar transacción

            // 1. Obtener URL de la foto antes de borrar (si existe)
            $sql_foto = "SELECT foto_url FROM candidatos WHERE id = ? AND id_eleccion = ?";
            $q_foto = $pdo->prepare($sql_foto);
            $q_foto->execute([$id_para_eliminar, $id_eleccion_actual]);
            $foto_data = $q_foto->fetch(PDO::FETCH_ASSOC);
            $foto_a_borrar = $foto_data ? $foto_data['foto_url'] : null;

            // 2. (Opcional pero recomendado) Verificar si hay votos asociados
            $sql_check_votos = "SELECT COUNT(*) FROM votos WHERE id_candidato = ?";
            $q_check = $pdo->prepare($sql_check_votos);
            $q_check->execute([$id_para_eliminar]);
            if ($q_check->fetchColumn() > 0) {
                 throw new Exception("No se puede eliminar el candidato porque tiene votos registrados.");
            }

            // 3. Eliminar de la BD
            $sql_delete = "DELETE FROM candidatos WHERE id = ? AND id_eleccion = ?";
            $q_delete = $pdo->prepare($sql_delete);
            $q_delete->execute([$id_para_eliminar, $id_eleccion_actual]);

            if ($q_delete->rowCount() > 0) {
                // 4. Intentar borrar el archivo físico de la foto si existía
                if ($foto_a_borrar && file_exists($foto_a_borrar)) {
                     if (!@unlink($foto_a_borrar)) { // Usar @ para suprimir warnings si falla, ya lo loggeamos
                         error_log("Error al borrar archivo de foto física: " . $foto_a_borrar);
                         // No crítico, el registro BD se borró.
                     }
                }
                $pdo->commit(); // Confirmar transacción
                header("Location: index.php?page=candidatos&success=deleted");
                exit;
            } else {
                 $pdo->rollBack(); // No se borró nada
                 $error_url_param = urlencode("No se encontró el candidato para eliminar (ID: $id_para_eliminar).");
            }
        } catch (PDOException $e) {
            $pdo->rollBack();
            error_log("Error PDO al eliminar candidato: " . $e->getMessage());
            $error_url_param = urlencode("Error de base de datos al eliminar.");
        } catch (Exception $e) { // Capturar la excepción de votos asociados
             $pdo->rollBack();
             error_log("Error al eliminar candidato: " . $e->getMessage());
             $error_url_param = urlencode($e->getMessage());
        }
    } else {
        $error_url_param = urlencode("ID inválido para eliminar.");
    }
    // Redirigir con error si algo falló
    header("Location: index.php?page=candidatos&error=" . $error_url_param);
    exit;
}


// --- Procesar Envío del Formulario (Crear o Actualizar Candidato) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['guardar_candidato'])) {
    $id_candidato_form = filter_input(INPUT_POST, 'id_candidato', FILTER_VALIDATE_INT); // ID si es actualización
    $nombre = trim($_POST['nombre'] ?? '');
    $apellido = trim($_POST['apellido'] ?? '');
    $partido = trim($_POST['partido_politico'] ?? '');
    $foto_actual = $_POST['foto_actual'] ?? null; // Foto existente si es update
    $foto_url_final = $foto_actual; // Por defecto mantenemos la actual

    // Validaciones básicas
    if (empty($nombre) || empty($apellido)) {
        $mensaje_error = "El nombre y el apellido son obligatorios.";
        // Recargar datos si estábamos editando para no perderlos visualmente
        if ($id_candidato_form) { $modo_edicion = true; $candidato_a_editar = $_POST; $candidato_a_editar['id'] = $id_candidato_form; }
    } else {
        // --- Manejo de Subida de Foto (Solo si se proporciona una nueva) ---
        $nueva_foto_subida = false;
        if (isset($_FILES['foto']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
            $foto_info = $_FILES['foto'];
            $permitidos = ['jpg', 'jpeg', 'png', 'gif'];
            $extension = strtolower(pathinfo($foto_info['name'], PATHINFO_EXTENSION));
            if (in_array($extension, $permitidos) && $foto_info['size'] <= 2 * 1024 * 1024) { // Max 2MB
                 $nombre_foto = uniqid('candidato_', true) . '.' . $extension;
                 $ruta_destino = UPLOADS_DIR . '/' . $nombre_foto;
                 if (!is_dir(UPLOADS_DIR)) { if(!@mkdir(UPLOADS_DIR, 0775, true)) { $mensaje_error="Error al crear directorio de subida."; }; }

                 if (empty($mensaje_error) && move_uploaded_file($foto_info['tmp_name'], $ruta_destino)) {
                    $foto_url_final = 'uploads/candidatos/' . $nombre_foto; // Nueva foto
                    $nueva_foto_subida = true;
                 } elseif(empty($mensaje_error)) { $mensaje_error = "Error al mover la nueva imagen subida."; }
            } else { $mensaje_error = "Archivo de foto nuevo no permitido o demasiado grande (Max 2MB: jpg, jpeg, png, gif)."; }
        } elseif (isset($_FILES['foto']) && $_FILES['foto']['error'] !== UPLOAD_ERR_NO_FILE) {
             $mensaje_error = "Error al subir la nueva foto (Código: " . $_FILES['foto']['error'] . ").";
        }
        // --- Fin Manejo de Foto ---

        // Proceder a guardar/actualizar en BD si no hubo errores de validación o subida
        if (empty($mensaje_error)) {
            try {
                if ($id_candidato_form) { // --- ACTUALIZAR ---
                    $sql = "UPDATE candidatos SET nombre = ?, apellido = ?, partido_politico = ?, foto_url = ?, updated_at = NOW()
                            WHERE id = ? AND id_eleccion = ?";
                    $q = $pdo->prepare($sql);
                    $q->execute([$nombre, $apellido, $partido ?: null, $foto_url_final, $id_candidato_form, $id_eleccion_actual]);

                    // Borrar foto antigua si se subió una nueva y la antigua existía y es diferente
                    if ($nueva_foto_subida && $foto_actual && $foto_actual != $foto_url_final && file_exists($foto_actual)) {
                        @unlink($foto_actual);
                    }
                    header("Location: index.php?page=candidatos&success=updated");
                    exit;

                } else { // --- CREAR ---
                    $sql = "INSERT INTO candidatos (id_eleccion, nombre, apellido, partido_politico, foto_url, created_at, updated_at)
                            VALUES (?, ?, ?, ?, ?, NOW(), NOW())";
                    $q = $pdo->prepare($sql);
                    $q->execute([$id_eleccion_actual, $nombre, $apellido, $partido ?: null, $foto_url_final]);
                    header("Location: index.php?page=candidatos&success=created");
                    exit;
                }

            } catch (PDOException $e) {
                error_log("Error DB candidato: " . $e->getMessage());
                $mensaje_error = "Error al " . ($id_candidato_form ? 'actualizar' : 'guardar') . " el candidato en la base de datos.";
                // Recargar datos si estábamos editando para no perderlos visualmente
                if ($id_candidato_form) { $modo_edicion = true; $candidato_a_editar = $_POST; $candidato_a_editar['id'] = $id_candidato_form;}
            }
        } else {
             // Si hubo error de validación/subida, asegurar que seguimos en modo edición si aplica
             if ($id_candidato_form) { $modo_edicion = true; $candidato_a_editar = $_POST; $candidato_a_editar['id'] = $id_candidato_form;}
        }
    }
}


// --- Cargar Lista de Candidatos (Leer) ---
// Se ejecuta siempre, después de cualquier operación POST
$lista_candidatos = [];
$error_carga_lista = null;
try {
    $sql_select = "SELECT id, nombre, apellido, partido_politico, foto_url FROM candidatos WHERE id_eleccion = ? ORDER BY apellido, nombre";
    $q_select = $pdo->prepare($sql_select);
    $q_select->execute([$id_eleccion_actual]);
    $lista_candidatos = $q_select->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Error al cargar lista candidatos: " . $e->getMessage());
    $error_carga_lista = "Error al cargar la lista de candidatos.";
}

// *** HTML ***
?>

<div class="container-fluid">
    <h1 class="mb-4"><i class="fas fa-users"></i> Gestión de Candidatos (Elección ID: <?php echo $id_eleccion_actual; ?>)</h1>

    <!-- Área de mensajes -->
    <?php if ($mensaje_exito): ?><div class="alert alert-success alert-dismissible fade show" role="alert"><?php echo htmlspecialchars($mensaje_exito); ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div><?php endif; ?>
    <?php if ($mensaje_error): ?><div class="alert alert-danger alert-dismissible fade show" role="alert"><?php echo htmlspecialchars($mensaje_error); ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div><?php endif; ?>
    <?php if ($error_carga_lista): ?><div class="alert alert-warning"><?php echo htmlspecialchars($error_carga_lista); ?></div><?php endif; ?>


    <!-- Formulario para agregar/editar -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary"><?php echo $modo_edicion ? 'Editar Candidato' : 'Registrar Nuevo Candidato'; ?></h6>
        </div>
        <div class="card-body">
            <!-- Acción apunta a la URL base de candidatos, puede incluir edit_id si estamos editando -->
            <form action="index.php?page=candidatos<?php echo $modo_edicion ? '&edit_id='.htmlspecialchars($candidato_a_editar['id']) : ''; ?>" method="POST" enctype="multipart/form-data">
                <!-- Campo oculto para ID en modo edición -->
                <?php if ($modo_edicion): ?>
                    <input type="hidden" name="id_candidato" value="<?php echo htmlspecialchars($candidato_a_editar['id']); ?>">
                    <input type="hidden" name="foto_actual" value="<?php echo htmlspecialchars($candidato_a_editar['foto_url'] ?? ''); ?>">
                <?php endif; ?>

                <div class="row mb-3">
                    <div class="col-md-6">
                        <label for="nombre" class="form-label">Nombre(s): <span class="text-danger">*</span></label>
                        <input type="text" class="form-control <?php echo (!empty($mensaje_error) && empty($_POST['nombre'] ?? '')) ? 'is-invalid' : ''; ?>" id="nombre" name="nombre" required value="<?php echo htmlspecialchars($candidato_a_editar['nombre'] ?? ($_POST['nombre'] ?? '')); ?>">
                    </div>
                    <div class="col-md-6">
                        <label for="apellido" class="form-label">Apellido(s): <span class="text-danger">*</span></label>
                        <input type="text" class="form-control <?php echo (!empty($mensaje_error) && empty($_POST['apellido'] ?? '')) ? 'is-invalid' : ''; ?>" id="apellido" name="apellido" required value="<?php echo htmlspecialchars($candidato_a_editar['apellido'] ?? ($_POST['apellido'] ?? '')); ?>">
                    </div>
                </div>
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label for="partido_politico" class="form-label">Partido Político:</label>
                        <input type="text" class="form-control" id="partido_politico" name="partido_politico" placeholder="Dejar vacío si es independiente" value="<?php echo htmlspecialchars($candidato_a_editar['partido_politico'] ?? ($_POST['partido_politico'] ?? '')); ?>">
                    </div>
                    <div class="col-md-6">
                        <label for="foto" class="form-label">Foto del Candidato <?php echo $modo_edicion ? '(Opcional: reemplazar actual)' : ''; ?>:</label>
                        <input class="form-control <?php echo (strpos($mensaje_error, 'foto') !== false) ? 'is-invalid' : ''; ?>" type="file" id="foto" name="foto" accept="image/png, image/jpeg, image/gif">
                        <small class="form-text text-muted">Formatos: PNG, JPG, GIF. Máx: 2MB.</small>
                         <?php if ($modo_edicion && !empty($candidato_a_editar['foto_url'])): ?>
                            <div class="mt-2">
                                <small>Foto actual:</small><br>
                                <img src="<?php echo htmlspecialchars($candidato_a_editar['foto_url']); ?>" alt="Foto actual" height="60" class="img-thumbnail" onerror="this.style.display='none'">
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                <input type="hidden" name="guardar_candidato" value="1">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-<?php echo $modo_edicion ? 'sync-alt' : 'save'; ?>"></i> <?php echo $modo_edicion ? 'Actualizar' : 'Guardar'; ?> Candidato
                </button>
                <!-- Botón Cancelar solo en modo edición -->
                <?php if ($modo_edicion): ?>
                <a href="index.php?page=candidatos" class="btn btn-secondary">
                    <i class="fas fa-times"></i> Cancelar Edición
                </a>
                <?php else: ?>
                <button type="reset" class="btn btn-secondary"><i class="fas fa-eraser"></i> Limpiar</button>
                <?php endif; ?>
            </form>
        </div>
    </div>

    <!-- Tabla para listar candidatos -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">Lista de Candidatos Registrados</h6>
        </div>
        <div class="card-body">
            <?php if (empty($lista_candidatos) && empty($error_carga_lista)): ?>
                <p class="text-center">No hay candidatos registrados para esta elección.</p>
            <?php elseif (!empty($lista_candidatos)): ?>
                <div class="table-responsive">
                    <table class="table table-bordered table-hover table-sm" id="dataTableCandidatos" width="100%" cellspacing="0">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th class="text-center">Foto</th>
                                <th>Nombre Completo</th>
                                <th>Partido Político</th>
                                <th class="text-center">Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($lista_candidatos as $candidato):
                                $nombre_completo_tabla = htmlspecialchars($candidato['nombre'] . ' ' . $candidato['apellido']);
                            ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($candidato['id']); ?></td>
                                    <td class="text-center align-middle">
                                        <img src="<?php echo !empty($candidato['foto_url']) ? htmlspecialchars($candidato['foto_url']) : 'assets/placeholder.png'; ?>"
                                             alt="Foto" width="50" height="50" style="object-fit: cover;" class="img-thumbnail"
                                             onerror="this.onerror=null; this.src='assets/placeholder.png';">
                                    </td>
                                    <td class="align-middle"><?php echo $nombre_completo_tabla; ?></td>
                                    <td class="align-middle"><?php echo htmlspecialchars($candidato['partido_politico'] ?: 'Independiente'); ?></td>
                                    <td class="text-center align-middle">
                                        <a href="index.php?page=candidatos&edit_id=<?php echo htmlspecialchars($candidato['id']); ?>" class="btn btn-sm btn-warning me-1" title="Editar">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <button type="button" class="btn btn-sm btn-danger" title="Eliminar"
                                                data-bs-toggle="modal" data-bs-target="#confirmDeleteModal"
                                                data-id="<?php echo htmlspecialchars($candidato['id']); ?>"
                                                data-name="<?php echo $nombre_completo_tabla; ?>"
                                                data-type="candidato"
                                                data-page="candidatos">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div> <!-- Fin container-fluid -->