<?php
// --- Script para Exportar Reporte de Votos (MEJORADO PARA EXCEL) ---

require_once 'config.php';
require_once 'includes/database.php';

// --- Validación de Parámetros ---
$format = $_GET['format'] ?? 'csv'; // Formato por defecto CSV
$id_eleccion_export = filter_input(INPUT_GET, 'election_id', FILTER_VALIDATE_INT);

if (!$id_eleccion_export) {
    die("Error: ID de elección inválido o no proporcionado.");
}

if ($format !== 'csv') {
    die("Error: Formato de exportación no soportado ('$format'). Solo 'csv' está disponible.");
}

// --- Variables ---
$resultados_export = [];
$total_general_export = 0;
$nombre_eleccion = "Eleccion ID " . $id_eleccion_export; // Nombre por defecto

// --- Conexión y Obtención de Datos (Lógica DUPLICADA de reporte_votos.php) ---
try {
    $pdo = Database::connect();

    // Opcional: Obtener nombre real de la elección
    $sql_elec = "SELECT nombre FROM elecciones WHERE id = ? LIMIT 1";
    $q_elec = $pdo->prepare($sql_elec);
    $q_elec->execute([$id_eleccion_export]);
    $eleccion_info = $q_elec->fetch(PDO::FETCH_ASSOC);
    if ($eleccion_info) {
        $nombre_eleccion = $eleccion_info['nombre'];
    }

    // Consulta principal para obtener resultados agregados
    $sql_resultados = "SELECT
                            c.nombre, c.apellido, c.partido_politico,
                            SUM(v.cantidad_votos) AS total_votos
                        FROM votos AS v
                        JOIN candidatos AS c ON v.id_candidato = c.id
                        WHERE v.id_eleccion = ?
                        GROUP BY c.id, c.nombre, c.apellido, c.partido_politico -- Asegurar agrupación correcta
                        ORDER BY total_votos DESC";

    $q_resultados = $pdo->prepare($sql_resultados);
    $q_resultados->execute([$id_eleccion_export]);
    $resultados_db = $q_resultados->fetchAll(PDO::FETCH_ASSOC);

    if (empty($resultados_db)) {
        Database::disconnect();
        die("No hay votos registrados para la elección ID $id_eleccion_export para exportar.");
    }

    $resultados_export = $resultados_db;

    // Calcular total general
    foreach ($resultados_export as $resultado) {
        $total_general_export += $resultado['total_votos'];
    }

    Database::disconnect();

} catch (PDOException $e) {
    error_log("Error PDO en export_report.php: " . $e->getMessage());
    if(isset($pdo)) { Database::disconnect(); }
    die("Error al obtener los datos para la exportación.");
} catch (Exception $e) {
     error_log("Error general en export_report.php: " . $e->getMessage());
     if(isset($pdo)) { Database::disconnect(); }
     die("Ocurrió un error inesperado durante la exportación.");
}

// --- Generación del CSV MEJORADO ---

// Nombre del archivo dinámico
$filename = "reporte_votos_" . preg_replace('/[^a-z0-9_-]/i', '', str_replace(' ', '_', $nombre_eleccion)) . "_" . date('Ymd_His') . ".csv";

// Encabezados HTTP para forzar descarga
header('Content-Type: text/csv; charset=utf-8'); // Mantenemos UTF-8
header('Content-Disposition: attachment; filename="' . $filename . '"');
header('Pragma: no-cache');
header('Expires: 0');

// Abrir el flujo de salida de PHP
$output = fopen('php://output', 'w');
if ($output === false) {
    die("Error al abrir el flujo de salida para generar el CSV.");
}

// *** MEJORA 1: Añadir BOM UTF-8 para Excel ***
fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));

// Definir el delimitador
$delimiter = ';'; // *** MEJORA 2: Usar punto y coma como delimitador ***

// Escribir Título e Información (usando el delimitador correcto)
fputcsv($output, ['Reporte de Votos'], $delimiter);
fputcsv($output, ['Elección:', $nombre_eleccion], $delimiter);
fputcsv($output, ['Fecha de Exportación:', date('d/m/Y H:i:s')], $delimiter);
fputcsv($output, [], $delimiter); // Línea vacía como separador

// Escribir la fila de encabezado de la tabla (usando el delimitador)
fputcsv($output, [
    'Candidato',
    'Partido Político',
    'Votos Totales',
    'Porcentaje (%)' // Mantener el símbolo % en el encabezado
], $delimiter);

// Escribir las filas de datos
foreach ($resultados_export as $resultado) {
    $porcentaje = ($total_general_export > 0)
                  ? round(($resultado['total_votos'] / $total_general_export) * 100, 2) // Calcular porcentaje
                  : 0;
    $nombre_completo_export = $resultado['nombre'] . ' ' . $resultado['apellido'];

    // Asegurar que los números no tengan formato específico de miles o decimales que confundan
    $votos_num = (int) $resultado['total_votos']; // Exportar como número entero
    // $porcentaje_num = $porcentaje; // Exportar el número directamente

    // *** MEJORA 3 (Alternativa): Usar punto como decimal para porcentaje si aún da problemas ***
    // $porcentaje_num_str = number_format($porcentaje, 2, '.', '');

    fputcsv($output, [
        $nombre_completo_export,
        $resultado['partido_politico'] ?: 'Independiente',
        $votos_num, // Votos como número
        $porcentaje // Porcentaje como número (Excel debería formatearlo)
        // O si usas la alternativa 3: $porcentaje_num_str
    ], $delimiter); // Usar el delimitador
}

// Escribir la fila de totales
fputcsv($output, [], $delimiter); // Separador
fputcsv($output, [
    'TOTAL',
    '', // Columna Partido vacía
    $total_general_export, // Total votos como número
    100 // Total porcentaje como número
], $delimiter); // Usar el delimitador

// fclose($output); // No estrictamente necesario para php://output

exit; // Detener el script

?>